#!/usr/bin/env python
# encoding: utf8
"""
Usage:
    spack env activate --sh > activate-ebrains.sh
    python module2view.py activate-ebrains.sh /path/to/install/root /path/to/view >activate-ebrains-view.sh

Spack creates a `source`-able shell script with path modifications for each
loaded package pointing to each individual packages install path. If a view
exists, these paths can be re-written to target the paths linked in the view.
The view-targeted shell script can then be converted to a module file.

LMOD [1]
    $LMOD_DIR/sh_to_modulefile  ./activate-ebrains-view.sh > ebrains/23.09.lua

Environment-Modules (TCL) [2]
    module sh-to-mod bash ./activate-ebrains-view.sh >$MODULE_PREFIX/ebrains/23.09


[1]: https://lmod.readthedocs.io/en/latest/260_sh_to_modulefile.html
[2]: https://modules.readthedocs.io/en/latest/cookbook/source-script-in-modulefile.html#implementation
"""
import os
import re
import sys
import logging
from pathlib import Path
from collections import Counter

SYSTEMNAME = os.environ.get("SYSTEMNAME", "jurecadc")
logging.basicConfig(level=logging.DEBUG)
log = logging.getLogger()

class Rewriter:
    def __init__(self, installpath: Path, viewpath: Path):
        self._installpath = Path(installpath)
        self._viewpath = Path(viewpath)
        self._export_re = re.compile(r"^export (?P<key>\w+)=['\"]?(?P<value>.*?)['\"]?;\n")
        self._package_re = re.compile(r"(?P<package>\w[^/]+)-(?P<hash>[0-9a-z]{32})/(?P<relative>.*)")
        if not self._installpath.is_dir():
            raise ValueError(f"installpath {self._installpath} is not a directory")
        if not self._viewpath.is_dir():
            raise ValueError(f"viewpath {self._viewpath} is not a directory")

    @property
    def installpath(self):
        return self._installpath

    @property
    def viewpath(self):
        return self._viewpath

    def __call__(self, line: str) -> str:
        "Parse an input line and rewrite if necessary."

        export = self._export_re.match(line.strip("'"))
        if export:
            key, value = export.groups()
            log.debug("found export of %s", key)

            if ":" in value:
                value = self._rewrite(value, separator=":")
            else:
                value = self._rewrite(value, separator=" ")
            return f"export {key}='{value[:]}'\n"
        else:
            log.debug("paste")
            return line

    def _rewrite(self, value: str, separator=":") -> str:
        """
        Change all spack install paths to the given viewpath and de-duplicate.

        Since not all variable values are collon-separated `:` like `$PATH`, the
        separator can optionally be changed (e.g. for `TCLLIBPATH`).
        """
        log.debug("REWRITE value split by '%s'", separator)
        log.debug("REWRITE value '%s'", value)

        if separator not in value:
            log.debug("value paste (no separator)")
            return value

        inpaths = value.split(separator)
        log.debug("inspecting %s paths", len(inpaths))

        outpaths = []
        for path in value.split(separator):
            if not path.startswith(str(self.installpath)):
                if path in outpaths:
                    log.debug("already in output: %s", path)
                    continue
                log.debug("append unchanged:  %s", path)
                outpaths.append(path)
                continue
            subpath = path[len(str(self.installpath)) :].lstrip("/")  # match tail after installpath
            log.debug(f"matching subpath={subpath}")
            toolpath = self._package_re.match(subpath)
            if toolpath:
                newpath = str(self._viewpath / toolpath.group("relative"))
                if newpath not in outpaths:
                    log.debug("append   changed:  %s", newpath)
                    outpaths.append(newpath)
                else:
                    log.debug("already in output: %s", newpath)
            else:
                raise ValueError(f"could not decompose path {subpath}")
        return separator.join(outpaths)


def main():
    """
    Convert all shell exports with paths to a Spack install directory to paths
    into an environment view.
    """
    # example installpaths
    # /p/usersoftware/swmanage/EBRAINS/{SYSTEMNAME}/install/linux-rocky8-zen2/gcc-11.3.0/py-setuptools-62.6.0-idjjqlikk266jeus6yoziulnifmnsjsi/lib/python3.10/site-packages

    shellname = Path(sys.argv[1])
    installpath = Path(sys.argv[2])
    viewpath = Path(sys.argv[3])

    rewrite = Rewriter(installpath=installpath, viewpath=viewpath)
    with shellname.open("r", encoding="utf8") as infile:
        for line in infile:
            sys.stdout.write(rewrite(line))


if __name__ == "__main__":
    main()
